<?php
require_once "db.php";
header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(["status" => "error", "message" => "Unauthorized"]);
    exit;
}

$conn->begin_transaction();

try {

    /* ===============================
       1. COLLECT DATA
    =============================== */

    $name        = trim($_POST['name'] ?? '');
    $location    = trim($_POST['location'] ?? '');
    $type        = $_POST['property_type'] ?? '';
    $description = trim($_POST['description'] ?? '');

    $bedrooms  = (int)($_POST['bedrooms'] ?? 0);
    $bathrooms = (int)($_POST['bathrooms'] ?? 0);

    $l_name  = trim($_POST['landlord_name'] ?? '');
    $l_phone = trim($_POST['landlord_phone'] ?? '');
    $l_email = trim($_POST['landlord_email'] ?? '');

    $units_count = (int)($_POST['units_count'] ?? 0);
    $unit_types  = $_POST['unit_types'] ?? '[]';

    /* ===============================
       2. VALIDATION
    =============================== */

    if (!$name || !$location || !$type) {
        throw new Exception("Required property fields are missing.");
    }

    if (!$l_name || !$l_phone || !$l_email) {
        throw new Exception("Landlord details are required.");
    }

    if (empty($_FILES['images']['name'][0])) {
        throw new Exception("At least one image is required.");
    }

    /* ===============================
       3. LANDLORD (REUSE OR CREATE)
    =============================== */

    $stmt = $conn->prepare("SELECT id FROM landlords WHERE email = ? OR phone = ? LIMIT 1");
    $stmt->bind_param("ss", $l_email, $l_phone);
    $stmt->execute();
    $res = $stmt->get_result();

    if ($res->num_rows > 0) {
        $landlord_id = $res->fetch_assoc()['id'];
    } else {
        $stmt = $conn->prepare("INSERT INTO landlords (name, phone, email) VALUES (?,?,?)");
        $stmt->bind_param("sss", $l_name, $l_phone, $l_email);
        $stmt->execute();
        $landlord_id = $conn->insert_id;
    }

    /* ===============================
       4. INSERT PROPERTY
    =============================== */

    $code = strtoupper(substr(md5(time() . $name), 0, 8));

    $stmt = $conn->prepare("
        INSERT INTO properties
        (landlord_id, name, code, location, property_type, description, bedrooms, bathrooms)
        VALUES (?,?,?,?,?,?,?,?)
    ");

    $stmt->bind_param(
        "isssssii",
        $landlord_id,
        $name,
        $code,
        $location,
        $type,
        $description,
        $bedrooms,
        $bathrooms
    );

    $stmt->execute();
    $property_id = $conn->insert_id;

    /* ===============================
       5. APARTMENT UNIT DATA
    =============================== */

    if ($type === "apartment" && $units_count > 0) {

        $stmt = $conn->prepare("
            INSERT INTO property_units
            (property_id, unit_type, units_count, vacant_count)
            VALUES (?,?,?,?)
        ");

        $stmt->bind_param(
            "isii",
            $property_id,
            $unit_types,
            $units_count,
            $units_count
        );

        $stmt->execute();
    }

    /* ===============================
       6. IMAGE UPLOADS
    =============================== */

    $uploadDir = "uploads/properties/";

    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }

    $mainImagePath = "";

    foreach ($_FILES['images']['tmp_name'] as $key => $tmpName) {

        if (!is_uploaded_file($tmpName)) continue;

        $ext = strtolower(pathinfo($_FILES['images']['name'][$key], PATHINFO_EXTENSION));

        $fileName = "prop_" . $property_id . "_" . uniqid() . "." . $ext;
        $targetPath = $uploadDir . $fileName;

        if (move_uploaded_file($tmpName, $targetPath)) {

            if ($key === 0) {
                $mainImagePath = $targetPath;
            }

            $imgStmt = $conn->prepare("
                INSERT INTO property_images (property_id, image_path)
                VALUES (?,?)
            ");
            $imgStmt->bind_param("is", $property_id, $targetPath);
            $imgStmt->execute();
        }
    }

    /* ===============================
       7. SET MAIN IMAGE
    =============================== */

    if ($mainImagePath) {
        $stmt = $conn->prepare("UPDATE properties SET main_image = ? WHERE id = ?");
        $stmt->bind_param("si", $mainImagePath, $property_id);
        $stmt->execute();
    }

    /* ===============================
       DONE
    =============================== */

    $conn->commit();

    echo json_encode([
        "status" => "success",
        "message" => "Property saved successfully"
    ]);

} catch (Exception $e) {

    $conn->rollback();

    echo json_encode([
        "status" => "error",
        "message" => $e->getMessage()
    ]);
}
