<?php
require_once "db.php";

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(["status"=>"error","message"=>"Unauthorized"]);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(["status"=>"error","message"=>"Invalid request"]);
    exit;
}

/* ================== SANITIZE INPUT ================== */
$property_id  = (int)($_POST['property_id'] ?? 0);
$name         = trim($_POST['name'] ?? '');
$location     = trim($_POST['location'] ?? '');
$type         = trim($_POST['property_type'] ?? '');
$description  = trim($_POST['description'] ?? '');
$units_count  = (int)($_POST['units_count'] ?? 0);
$vacant_count = (int)($_POST['vacant_count'] ?? 0);

/* LANDLORD */
$landlord_id  = (int)($_POST['landlord_id'] ?? 0);
$l_name       = trim($_POST['l_name'] ?? '');
$l_phone      = trim($_POST['l_phone'] ?? '');
$l_email      = trim($_POST['l_email'] ?? '');


/* ================== VALIDATION ================== */
if (
    $property_id <= 0 ||
    $name === '' ||
    $location === '' ||
    !in_array($type, ['single_house','apartment'])
) {
    echo json_encode(["status"=>"error","message"=>"Invalid or missing fields"]);
    exit;
}

if ($units_count < 0 || $vacant_count < 0) {
    echo json_encode(["status"=>"error","message"=>"Invalid unit numbers"]);
    exit;
}

if ($vacant_count > $units_count) {
    echo json_encode(["status"=>"error","message"=>"Vacancy exceeds total units"]);
    exit;
}


/* ================== START TRANSACTION ================== */
$conn->begin_transaction();

try {

    /* =====================================================
       1️⃣ HANDLE LANDLORD (CREATE OR UPDATE)
    ====================================================== */
    if ($l_name !== '' && $l_phone !== '' && $l_email !== '') {

        if ($landlord_id > 0) {

            // Update existing landlord
            $stmt = $conn->prepare("
                UPDATE landlords
                SET name=?, phone=?, email=?
                WHERE id=?
            ");
            $stmt->bind_param("sssi", $l_name, $l_phone, $l_email, $landlord_id);

            if (!$stmt->execute()) {
                throw new Exception("Failed to update landlord");
            }

            $stmt->close();

        } else {

            // Create new landlord
            $stmt = $conn->prepare("
                INSERT INTO landlords (name, phone, email)
                VALUES (?,?,?)
            ");
            $stmt->bind_param("sss", $l_name, $l_phone, $l_email);

            if (!$stmt->execute()) {
                throw new Exception("Failed to create landlord");
            }

            $landlord_id = $stmt->insert_id;
            $stmt->close();
        }

        // Attach landlord to property
        $stmt = $conn->prepare("
            UPDATE properties
            SET landlord_id=?
            WHERE id=?
        ");
        $stmt->bind_param("ii", $landlord_id, $property_id);

        if (!$stmt->execute()) {
            throw new Exception("Failed to attach landlord");
        }

        $stmt->close();
    }


    /* =====================================================
       2️⃣ UPDATE PROPERTY DETAILS
    ====================================================== */
    $stmt = $conn->prepare("
        UPDATE properties
        SET name=?, location=?, property_type=?, description=?
        WHERE id=?
    ");

    $stmt->bind_param(
        "ssssi",
        $name,
        $location,
        $type,
        $description,
        $property_id
    );

    if (!$stmt->execute()) {
        throw new Exception("Failed to update property");
    }

    $stmt->close();


    /* =====================================================
       3️⃣ UPSERT PROPERTY UNITS
    ====================================================== */
    $check = $conn->prepare("
        SELECT id FROM property_units WHERE property_id=?
    ");
    $check->bind_param("i", $property_id);
    $check->execute();
    $result = $check->get_result();
    $exists = $result->num_rows > 0;
    $check->close();

    if ($exists) {

        $stmt = $conn->prepare("
            UPDATE property_units
            SET units_count=?, vacant_count=?
            WHERE property_id=?
        ");
        $stmt->bind_param("iii", $units_count, $vacant_count, $property_id);

        if (!$stmt->execute()) {
            throw new Exception("Failed to update units");
        }

        $stmt->close();

    } else {

        $stmt = $conn->prepare("
            INSERT INTO property_units (property_id, units_count, vacant_count)
            VALUES (?,?,?)
        ");
        $stmt->bind_param("iii", $property_id, $units_count, $vacant_count);

        if (!$stmt->execute()) {
            throw new Exception("Failed to insert units");
        }

        $stmt->close();
    }


    /* ================== COMMIT ================== */
    $conn->commit();

    echo json_encode([
        "status"  => "success",
        "message" => "Property updated successfully"
    ]);

} catch (Exception $e) {

    $conn->rollback();

    echo json_encode([
        "status"  => "error",
        "message" => $e->getMessage()
    ]);
}
