<?php
// 🔹 Enable errors for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once "db.php";

// 🔹 Ensure session is started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// 🔹 If already logged in, redirect based on role
if (isset($_SESSION['user_id']) && isset($_SESSION['login_time']) && (time() - $_SESSION['login_time']) < 86400) {
    switch ($_SESSION['role']) {
        case 'admin': header("Location: admin_dashboard.php"); exit;
        case 'landlord': header("Location: landlord_dashboard.php"); exit;
        case 'user': header("Location: user_dashboard.php"); exit;
        default: header("Location: index.php"); exit;
    }
}

$error = "";

if ($_SERVER["REQUEST_METHOD"] === "POST") {
    $login    = trim($_POST["login"] ?? '');
    $password = $_POST["password"] ?? '';

    if ($login && $password) {
        $stmt = $conn->prepare("SELECT id, username, email, password, role, status FROM users WHERE username = ? OR email = ? LIMIT 1");
        if ($stmt) {
            $stmt->bind_param("ss", $login, $login);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result && $result->num_rows === 1) {
                $user = $result->fetch_assoc();
                if (password_verify($password, $user['password'])) {
                    if ($user['status'] === 'frozen') {
                        $error = "Access Denied. Contact Admin.";
                    } elseif ($user['status'] === 'deleted') {
                        $error = "Account no longer active.";
                    } else {
                        $_SESSION['user_id']    = $user['id'];
                        $_SESSION['username']   = $user['username'];
                        $_SESSION['role']       = $user['role'];
                        $_SESSION['login_time'] = time();

                        if ($user['role'] === 'landlord') {
                            $stmt2 = $conn->prepare("SELECT id FROM landlords WHERE email = ? LIMIT 1");
                            $stmt2->bind_param("s", $user['email']);
                            $stmt2->execute();
                            $landlord = $stmt2->get_result()->fetch_assoc();
                            if ($landlord) $_SESSION['landlord_id'] = $landlord['id'];
                            $stmt2->close();
                            header("Location: landlord_dashboard.php");
                        } else {
                            header("Location: " . ($user['role'] === 'admin' ? "admin_dashboard.php" : "user_dashboard.php"));
                        }
                        exit;
                    }
                } else { $error = "Invalid credentials."; }
            } else { $error = "Invalid credentials."; }
            $stmt->close();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Login | Lifeshed Realtors</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link rel="icon" type="image/png" href="images/Lifeshedlogo.png">
    
    <link href="https://fonts.googleapis.com/css?family=Poppins:400,600,700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="css/bootstrap.css">
    <link rel="stylesheet" href="css/style.css">

    <style>
        body { font-family: 'Poppins', sans-serif; background: #f4f7f6; margin: 0; }

        /* HERO / BACKGROUND SECTION */
        .login-hero {
            position: relative;
            background: linear-gradient(rgba(0, 31, 63, 0.85), rgba(0, 31, 63, 0.85)), url('images/hero-bg.jpg') center/cover;
            min-height: 80vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 100px 15px;
        }

        /* NAVBAR branding */
        .header_section { background-color: #001f3f; padding: 10px 0; }
        .navbar-brand img { width: 160px; }

        /* LOGIN CARD */
        .login-card {
            background: #ffffff;
            padding: 40px;
            border-radius: 15px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.2);
            border-bottom: 5px solid #FFD700;
        }

        .login-card h2 { color: #001f3f; font-weight: 700; text-align: center; margin-bottom: 30px; }
        .form-control { border-radius: 8px; padding: 12px; border: 1px solid #ddd; }
        
        .btn-login { 
            background: #001f3f; 
            color: #FFD700; 
            width: 100%; 
            padding: 12px; 
            font-weight: 600; 
            border: none;
            border-radius: 8px;
            margin-top: 10px;
        }
        .btn-login:hover { background: #FFD700; color: #001f3f; }

        .password-wrapper { position: relative; }
        .toggle-password { position: absolute; right: 15px; top: 12px; cursor: pointer; color: #666; }

        /* FOOTER BRANDING */
        .info_section { background-color: #001f3f; color: #ffffff; padding: 60px 0; }
        .info_section h5 { color: #FFD700; font-weight: 700; margin-bottom: 20px; }
        .footer_section { background: #001226; color: #fff; padding: 20px 0; }
        .error { background: #fff1f1; color: #d9534f; padding: 10px; border-radius: 5px; font-size: 14px; margin-bottom: 20px; text-align: center; border: 1px solid #d9534f; }
    </style>
</head>
<body>

    <header class="header_section">
        <div class="container">
            <nav class="navbar navbar-expand-lg custom_nav-container">
                <a class="navbar-brand" href="index.php">
                    <img src="images/Lifeshedlogo.png" alt="Lifeshed Realtors">
                </a>
            </nav>
        </div>
    </header>

    <div class="login-hero">
        <div class="login-card">
            <h2>Welcome Back</h2>

            <?php if($error): ?>
                <div class="error"><?= htmlspecialchars($error) ?></div>
            <?php endif; ?>

            <form method="POST" autocomplete="off">
                <div class="form-group">
                    <label>Username or Email</label>
                    <input name="login" class="form-control" placeholder="Enter your details" required>
                </div>
                
                <div class="form-group">
                    <label>Password</label>
                    <div class="password-wrapper">
                        <input id="password" name="password" type="password" class="form-control" placeholder="••••••••" required>
                        <span class="toggle-password" onclick="togglePassword()">👁️</span>
                    </div>
                </div>

                <div class="text-right mb-3">
                    <a href="forgot_password.php" style="color: #001f3f; font-size: 14px;">Forgot Password?</a>
                </div>

                <button type="submit" class="btn-login">Secure Login</button>
            </form>
        </div>
    </div>

    <section class="info_section">
        <div class="container">
            <div class="row">
                <div class="col-md-6">
                    <h5>Lifeshed Realtors</h5>
                    <p>📍 Highlands Mall, Eldoret (G7C8+94P)</p>
                    <p>📞 0725 608 170</p>
                    <p>✉️ info@lifeshedrealtors.co.ke</p>
                </div>
                <div class="col-md-6">
                    <h5>Quick Links</h5>
                    <ul class="list-unstyled">
                        <li><a href="index.php" class="text-white">Home</a></li>
                        <li><a href="about.html" class="text-white">The Story of Lifeshed</a></li>
                        <li><a href="house.php" class="text-white">House Listings</a></li>
                    </ul>
                </div>
            </div>
        </div>
    </section>

    <footer class="footer_section text-center">
        <div class="container">
            <p>&copy; <span id="displayYear"></span> Lifeshed Realtors. Professionalism & Integrity.</p>
        </div>
    </footer>

    <script>
        function togglePassword() {
            const p = document.getElementById('password');
            p.type = p.type === 'password' ? 'text' : 'password';
        }
        document.getElementById("displayYear").innerHTML = new Date().getFullYear();
    </script>
    <script src="js/jquery-3.4.1.min.js"></script>
    <script src="js/bootstrap.js"></script>

</body>
</html>