<?php
require_once "db.php";

/* =========================
   ACCESS CONTROL
========================= */
if (
    !isset($_SESSION['user_id']) ||
    !isset($_SESSION['role']) ||
    $_SESSION['role'] !== 'landlord' ||
    !isset($_SESSION['landlord_id'])
) {
    header("Location: login.php");
    exit;
}

$landlord_id = (int)$_SESSION['landlord_id'];

/* =========================
   FETCH LANDLORD DETAILS
========================= */
$stmt = $conn->prepare("
    SELECT name, email, phone
    FROM landlords
    WHERE id = ?
    LIMIT 1
");
$stmt->bind_param("i", $landlord_id);
$stmt->execute();
$landlord = $stmt->get_result()->fetch_assoc();
$stmt->close();

/* If landlord record missing, force logout */
if (!$landlord) {
    header("Location: logout.php");
    exit;
}

/* =========================
   FETCH PROPERTY STATS
========================= */

/* Total Properties */
$totalProperties = $conn->query("
    SELECT COUNT(*) as total
    FROM properties
    WHERE landlord_id = $landlord_id
")->fetch_assoc()['total'] ?? 0;

/* Total Units + Vacant */
$stats = $conn->query("
    SELECT 
        SUM(pu.units_count) as total_units,
        SUM(pu.vacant_count) as total_vacant
    FROM properties p
    LEFT JOIN property_units pu ON p.id = pu.property_id
    WHERE p.landlord_id = $landlord_id
")->fetch_assoc();

$totalUnits  = $stats['total_units'] ?? 0;
$totalVacant = $stats['total_vacant'] ?? 0;

/* =========================
   FETCH PROPERTIES LIST
========================= */
$properties = $conn->query("
    SELECT p.id, p.name, p.location, p.property_type,
           pu.units_count, pu.vacant_count
    FROM properties p
    LEFT JOIN property_units pu ON p.id = pu.property_id
    WHERE p.landlord_id = $landlord_id
");
?>

<!DOCTYPE html>
<html>
<head>
    <title>Landlord Dashboard | Lifeshed Realtors</title>
    <meta charset="UTF-8">
    <style>
        body {
            font-family: Arial, sans-serif;
            background:#f4f6f9;
            margin:0;
        }

        .header {
            background:#0f2a44;
            color:#fff;
            padding:20px;
            display:flex;
            justify-content:space-between;
            align-items:center;
        }

        .container {
            padding:30px;
        }

        .stats {
            display:grid;
            grid-template-columns:repeat(3, 1fr);
            gap:20px;
            margin-bottom:30px;
        }

        .card {
            background:#fff;
            padding:20px;
            border-radius:12px;
            box-shadow:0 4px 15px rgba(0,0,0,0.08);
        }

        .card h3 {
            margin:0;
            font-size:22px;
            color:#0f2a44;
        }

        .card p {
            margin:5px 0 0;
            color:#555;
        }

        table {
            width:100%;
            border-collapse:collapse;
            background:#fff;
            border-radius:12px;
            overflow:hidden;
        }

        th, td {
            padding:12px;
            text-align:left;
            border-bottom:1px solid #eee;
        }

        th {
            background:#f1f1f1;
        }

        .logout-btn {
            background:#fff;
            color:#0f2a44;
            padding:8px 15px;
            border-radius:8px;
            text-decoration:none;
            font-weight:bold;
        }

        .logout-btn:hover {
            background:#ddd;
        }
    </style>
</head>
<body>

<div class="header">
    <div>
        <h2>Welcome, <?= htmlspecialchars($landlord['name']) ?></h2>
        <small><?= htmlspecialchars($landlord['email']) ?></small>
    </div>
    <a href="logout.php" class="logout-btn">Logout</a>
</div>

<div class="container">

    <!-- STATS -->
    <div class="stats">

        <div class="card">
            <h3><?= (int)$totalProperties ?></h3>
            <p>Total Properties</p>
        </div>

        <div class="card">
            <h3><?= (int)$totalUnits ?></h3>
            <p>Total Units</p>
        </div>

        <div class="card">
            <h3><?= (int)$totalVacant ?></h3>
            <p>Vacant Units</p>
        </div>

    </div>

    <!-- PROPERTY TABLE -->
    <div class="card">
        <h3 style="margin-bottom:15px;">My Properties</h3>

        <table>
            <tr>
                <th>Name</th>
                <th>Location</th>
                <th>Type</th>
                <th>Total Units</th>
                <th>Vacant</th>
            </tr>

            <?php while($row = $properties->fetch_assoc()): ?>
            <tr>
                <td><?= htmlspecialchars($row['name']) ?></td>
                <td><?= htmlspecialchars($row['location']) ?></td>
                <td><?= ucfirst($row['property_type']) ?></td>
                <td><?= (int)$row['units_count'] ?></td>
                <td><?= (int)$row['vacant_count'] ?></td>
            </tr>
            <?php endwhile; ?>

        </table>

    </div>

</div>

</body>
</html>
