<?php
session_start();
require_once "db.php";

/* ======================================================
   SECURITY CHECK
======================================================*/
if (!isset($_SESSION['user_id'])) {
    http_response_code(403);
    exit("Unauthorized");
}

if (!isset($_POST['action'])) {
    http_response_code(400);
    exit("Invalid request");
}

$action = $_POST['action'];

/* ======================================================
   ADD USER
======================================================*/
if ($action === 'add') {

    $username = trim($_POST['username'] ?? '');
    $email    = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $role     = $_POST['role'] ?? '';

    // Validate inputs
    if ($username === '' || $email === '' || $password === '' || !in_array($role, ['admin','landlord','user'])) {
        http_response_code(400);
        exit("All fields are required and role must be valid.");
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        exit("Invalid email format.");
    }

    // Check duplicate username
    $stmt = $conn->prepare("SELECT id FROM users WHERE username=?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 0) {
        $stmt->close();
        exit("Username already exists.");
    }
    $stmt->close();

    // Check duplicate email
    $stmt = $conn->prepare("SELECT id FROM users WHERE email=?");
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 0) {
        $stmt->close();
        exit("Email already exists.");
    }
    $stmt->close();

    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

    $conn->begin_transaction();
    try {
        // Insert into users
        $stmt = $conn->prepare("INSERT INTO users (username,email,password,role,status,created_at) VALUES (?,?,?,?, 'active', NOW())");
        $stmt->bind_param("ssss", $username, $email, $hashedPassword, $role);
        if (!$stmt->execute()) {
            throw new Exception("Failed to create user.");
        }
        $userId = $stmt->insert_id;
        $stmt->close();

        // If landlord, create landlord record
        if ($role === 'landlord') {
            $stmt2 = $conn->prepare("INSERT INTO landlords (name, phone, email, user_id, created_at) VALUES (?,?,?, ?, NOW())");
            $emptyPhone = '';
            $stmt2->bind_param("sssi", $username, $emptyPhone, $email, $userId);
            if (!$stmt2->execute()) throw new Exception("Failed to create landlord record.");
            $stmt2->close();
        }

        $conn->commit();
        echo "success";
    } catch (Exception $e) {
        $conn->rollback();
        http_response_code(500);
        echo $e->getMessage();
    }
    exit;
}

/* ======================================================
   EDIT USER
======================================================*/
if ($action === 'edit') {
    $userId  = (int) ($_POST['user_id'] ?? 0);
    $username = trim($_POST['username'] ?? '');
    $email    = trim($_POST['email'] ?? '');
    $role     = $_POST['role'] ?? '';

    if ($userId <= 0 || $username === '' || $email === '' || !in_array($role, ['admin','landlord','user'])) {
        http_response_code(400);
        exit("Invalid data.");
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        exit("Invalid email format.");
    }

    // Check duplicates (excluding current user)
    $stmt = $conn->prepare("SELECT id FROM users WHERE username=? AND id != ?");
    $stmt->bind_param("si", $username, $userId);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 0) {
        $stmt->close();
        exit("Username already exists.");
    }
    $stmt->close();

    $stmt = $conn->prepare("SELECT id FROM users WHERE email=? AND id != ?");
    $stmt->bind_param("si", $email, $userId);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 0) {
        $stmt->close();
        exit("Email already exists.");
    }
    $stmt->close();

    $conn->begin_transaction();
    try {
        // Get current role
        $stmt = $conn->prepare("SELECT role FROM users WHERE id=?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $stmt->bind_result($currentRole);
        $stmt->fetch();
        $stmt->close();

        // Update user
        $stmt = $conn->prepare("UPDATE users SET username=?, email=?, role=? WHERE id=?");
        $stmt->bind_param("sssi", $username, $email, $role, $userId);
        if (!$stmt->execute()) throw new Exception("Failed to update user.");
        $stmt->close();

        // Handle landlord table sync
        if ($role === 'landlord' && $currentRole !== 'landlord') {
            $stmt2 = $conn->prepare("INSERT INTO landlords (name, phone, email, user_id, created_at) VALUES (?,?,?, ?, NOW())");
            $emptyPhone = '';
            $stmt2->bind_param("sssi", $username, $emptyPhone, $email, $userId);
            $stmt2->execute();
            $stmt2->close();
        } elseif ($role !== 'landlord' && $currentRole === 'landlord') {
            $stmt2 = $conn->prepare("DELETE FROM landlords WHERE user_id=?");
            $stmt2->bind_param("i", $userId);
            $stmt2->execute();
            $stmt2->close();
        }

        $conn->commit();
        echo "success";
    } catch (Exception $e) {
        $conn->rollback();
        http_response_code(500);
        echo $e->getMessage();
    }
    exit;
}

/* ======================================================
   STATUS ACTIONS: freeze, unfreeze, delete, activate
======================================================*/
$statusActions = [
    'freeze'   => 'frozen',
    'unfreeze' => 'active',
    'delete'   => 'deleted',
    'activate' => 'active'
];

if (isset($statusActions[$action])) {
    $userId = (int) ($_POST['user_id'] ?? 0);
    if ($userId <= 0) {
        http_response_code(400);
        exit("Invalid user ID.");
    }

    $status = $statusActions[$action];
    $stmt = $conn->prepare("UPDATE users SET status=? WHERE id=?");
    $stmt->bind_param("si", $status, $userId);
    if (!$stmt->execute()) {
        http_response_code(500);
        exit("Failed to update status.");
    }
    $stmt->close();
    echo "success";
    exit;
}

/* ======================================================
   UNKNOWN ACTION
======================================================*/
http_response_code(400);
exit("Invalid request.");
