<?php
// 🔹 Enable errors for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once "db.php";

// 🔹 Ensure session is started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// 🔹 If already logged in and session still valid, redirect
if (isset($_SESSION['user_id']) && isset($_SESSION['login_time']) && (time() - $_SESSION['login_time']) < 86400) {
    switch ($_SESSION['role']) {
        case 'admin':
            header("Location: admin_dashboard.php");
            exit;
        case 'landlord':
            header("Location: landlord_dashboard.php");
            exit;
        case 'user':
            header("Location: user_dashboard.php");
            exit;
        default:
            header("Location: index.php");
            exit;
    }
}

$error = "";

// 🔹 Handle login submission
if ($_SERVER["REQUEST_METHOD"] === "POST") {

    $login    = trim($_POST["login"] ?? '');
    $password = $_POST["password"] ?? '';

    if ($login && $password) {

        // ✅ Prepare query to find user by username OR email
        $stmt = $conn->prepare(
            "SELECT id, username, email, password, role, status
             FROM users
             WHERE username = ? OR email = ?
             LIMIT 1"
        );

        if ($stmt) {
            $stmt->bind_param("ss", $login, $login);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result && $result->num_rows === 1) {

                $user = $result->fetch_assoc();

                // 🔹 Verify password
                if (password_verify($password, $user['password'])) {

                    // 🔹 Check account status
                    if ($user['status'] === 'frozen') {
                        $error = "Access Denied. Please contact the administrator.";
                    } elseif ($user['status'] === 'deleted') {
                        $error = "Your account is no longer active.";
                    } else {
                        // 🔹 Set session
                        $_SESSION['user_id']    = $user['id'];
                        $_SESSION['username']   = $user['username'];
                        $_SESSION['role']       = $user['role'];
                        $_SESSION['login_time'] = time();

                        // 🔹 Role-based redirect
                        switch ($user['role']) {
                            case 'admin':
                                header("Location: admin_dashboard.php");
                                exit;

                            case 'landlord':
                                // Optional: fetch landlord ID from landlords table
                                $stmt2 = $conn->prepare("SELECT id FROM landlords WHERE email = ? LIMIT 1");
                                if ($stmt2) {
                                    $stmt2->bind_param("s", $user['email']);
                                    $stmt2->execute();
                                    $landlord = $stmt2->get_result()->fetch_assoc();
                                    $stmt2->close();
                                    if ($landlord) {
                                        $_SESSION['landlord_id'] = $landlord['id'];
                                    }
                                }
                                header("Location: landlord_dashboard.php");
                                exit;

                            case 'user':
                                header("Location: user_dashboard.php");
                                exit;

                            default:
                                header("Location: index.php");
                                exit;
                        }
                    }
                } else {
                    $error = "Invalid username/email or password.";
                }
            } else {
                $error = "Invalid username/email or password.";
            }
            $stmt->close();
        } else {
            $error = "Database error: " . $conn->error;
        }
    } else {
        $error = "Please fill in all fields.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Lifeshed Login</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://fonts.googleapis.com/css?family=Poppins:400,600,700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="css/bootstrap.css">
    <link rel="stylesheet" href="css/style.css">
    <style>
        body {font-family:'Poppins',sans-serif;background:#f4f6f9;}
        .login-wrapper {min-height:100vh;display:flex;justify-content:center;align-items:center;padding:60px 15px;}
        .login-card {background:#fff;padding:35px;border-radius:16px;width:100%;max-width:380px;box-shadow:0 15px 40px rgba(0,0,0,0.2);}
        .login-card h2 {text-align:center;color:#0f2a44;font-weight:600;margin-bottom:25px;}
        .login-card input {width:100%;padding:10px;margin-bottom:15px;border-radius:8px;border:1px solid #ccc;}
        .password-wrapper {position:relative;}
        .toggle-password {position:absolute;right:12px;top:50%;transform:translateY(-50%);cursor:pointer;font-size:18px;color:#555;}
        .login-card button {width:100%;padding:10px;border:none;border-radius:8px;background:#0f2a44;color:#fff;font-size:16px;cursor:pointer;}
        .error {background:#ffe5e5;color:#b30000;padding:10px;border-radius:8px;margin-bottom:15px;text-align:center;font-size:14px;}
    </style>
</head>
<body>

<div class="login-wrapper">
    <form class="login-card" method="POST" autocomplete="off">
        <h2>Lifeshed Realtors</h2>

        <?php if($error): ?>
            <div class="error"><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>

        <input name="login" placeholder="Username or Email" required>
        <div class="password-wrapper">
            <input id="password" name="password" type="password" placeholder="Password" required>
            <span class="toggle-password" onclick="togglePassword()">👁️</span>
        </div>

        <div class="forgot-link" style="text-align:right;margin-bottom:15px;font-size:14px;">
            <a href="forgot_password.php">Forgot Password?</a>
        </div>

        <button type="submit">Login</button>
    </form>
</div>

<script>
function togglePassword() {
    const p = document.getElementById('password');
    p.type = p.type === 'password' ? 'text' : 'password';
}
</script>
</body>
</html>
